<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Notification;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Helpers\TraitFeedHelper;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Exceptions\BreakException;
use Erlage\Photogram\Data\Tables\Post\PostTable;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Post\PostLikeTable;
use Erlage\Photogram\Data\Tables\Post\PostSaveTable;
use Erlage\Photogram\Data\Tables\User\UserFollowTable;
use Erlage\Photogram\Data\Dtos\Common\NotificationsCountDTO;
use Erlage\Photogram\Data\Models\Notification\NotificationEnum;
use Erlage\Photogram\Data\Models\Notification\NotificationModel;
use Erlage\Photogram\Data\Tables\Notification\NotificationTable;
use Erlage\Photogram\Data\Models\Notification\NotificationFinder;

final class NotificationContent extends ExceptionalRequests
{
    use TraitFeedHelper;

    public static function count(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            |  ensure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | initialize
            |--------------------------------------------------------------------------
            */

            $likeCount = 0;
            $commentCount = 0;
            $followCount = 0;
            $otherCount = 0;

            /*
            |--------------------------------------------------------------------------
            | if paused, don't fetch or send anything..
            | we still send 0zeros so that notifcations can fade away if user just
            | toggled pause all option
            |--------------------------------------------------------------------------
            */

            if ( ! self::$authedUserModel -> getMetaPushSettings() -> arePaused())
            {
                /*
                |--------------------------------------------------------------------------
                | find unread notifications
                |--------------------------------------------------------------------------
                */

                $notificationFinder = (new NotificationFinder())
                    -> setMetaIsRead(NotificationEnum::META_IS_READ_NO)
                    -> setToUserId(self::$authedUserModel -> getId())
                    -> limit('100') // hard-coded is better. makes no sense to control this
                    -> find();

                /*
                |--------------------------------------------------------------------------
                | if found unread, add them
                |--------------------------------------------------------------------------
                */

                if ($notificationFinder -> isFound())
                {
                    /*
                    |--------------------------------------------------------------------------
                    | do counting
                    |--------------------------------------------------------------------------
                    */

                    while ($notificationFinder -> canPop())
                    {
                        $notificationModel = $notificationFinder -> popModelFromResults();

                        switch ($notificationModel -> getMetaType())
                    {
                        // like

                        case NotificationEnum::META_TYPE_POST_LIKE:
                        case NotificationEnum::META_TYPE_POST_LIKE_ON_PHOTO_OF_YOU:
                        case NotificationEnum::META_TYPE_POST_COMMENT_LIKE:
                            $likeCount++;

                            break;

                        // comment

                        case NotificationEnum::META_TYPE_POST_COMMENT:
                        case NotificationEnum::META_TYPE_POST_COMMENT_ON_PHOTO_OF_YOU:
                        case NotificationEnum::META_TYPE_POST_COMMENT_REPLY:
                            $commentCount++;

                            break;

                        // user follow

                        case NotificationEnum::META_TYPE_USER_FOLLOW:
                        case NotificationEnum::META_TYPE_USER_FOLLOW_PENDING:
                        case NotificationEnum::META_TYPE_USER_FOLLOW_ACCEPTED:
                            $followCount++;

                            break;

                        // other

                        case NotificationEnum::META_TYPE_PHOTO_OF_YOU:
                            $otherCount++;

                            break;
                    }
                    }
                }
            }

            /*
            |--------------------------------------------------------------------------
            | prepare data for dispatch
            |--------------------------------------------------------------------------
            */

            $notificationsCountDTO = (new NotificationsCountDTO())
                -> setLikeCount((string) $likeCount)
                -> setCommentCount((string) $commentCount)
                -> setFollowCount((string) $followCount)
                -> setOtherCount((string) $otherCount);

            /*
            |--------------------------------------------------------------------------
            | add to response
            |--------------------------------------------------------------------------
            */

            self::$response -> setContent(
                NotificationsCountDTO::DTO_NAME,
                array($notificationsCountDTO -> jsonSerialize())
            );
        });
    }

    public static function load(string $loadType): void
    {
        self::feedHelperInit('', $loadType);

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $offset = self::$request -> findKeyOffset(NotificationTable::ID, NotificationTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | query builder
            |--------------------------------------------------------------------------
            */

            $notificationTableQuery = (new Query()) -> from(NotificationTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | for current user
            |--------------------------------------------------------------------------
            */

            $notificationTableQuery -> where(NotificationTable::TO_USER_ID, self::$authedUserModel -> getId());

            /*
            |--------------------------------------------------------------------------
            | selection order
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingLatestContent())
            {
                $notificationTableQuery -> greaterThan(NotificationTable::ID, $offset);
            }
            else
            {
                $notificationTableQuery -> lessThan(NotificationTable::ID, $offset);
            }

            /*
            |--------------------------------------------------------------------------
            | order by & limit
            |--------------------------------------------------------------------------
            */

            $notificationTableQuery
                -> orderByDesc(NotificationTable::ID)
                -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_NOTIFICATION));

            /*
            |--------------------------------------------------------------------------
            | get beans
            |--------------------------------------------------------------------------
            */

            $notificationBeans = $notificationTableQuery -> select();

            /*
            |--------------------------------------------------------------------------
            | check end of results
            |--------------------------------------------------------------------------
            */

            if (0 == \count($notificationBeans))
            {
                return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | prepare maps
            |--------------------------------------------------------------------------
            */

            self::processBeans(NotificationTable::getTableName(), $notificationBeans, function (NotificationModel $model)
            {
                $linkedContent = $model -> getLinkedContent();

                if ($linkedContent -> containUserIds())
                {
                    $userIdsToFetch = $linkedContent -> getUserIds() -> getAll();

                    if (\count($userIdsToFetch) > 3)
                    {
                        $userIdsToFetch = \array_splice($userIdsToFetch, -3, 3);
                    }

                    foreach ($userIdsToFetch as $id)
                    {
                        self::addDependency(UserTable::getTableName(), $id);
                    }
                }

                if ($linkedContent -> containPostIds())
                {
                    $postIdsToFetch = $linkedContent -> getPostIds() -> getAll();

                    if (\count($postIdsToFetch) > 3)
                    {
                        $postIdsToFetch = \array_splice($postIdsToFetch, -3, 3);
                    }

                    foreach ($postIdsToFetch as $id)
                    {
                        self::addDependency(PostTable::getTableName(), $id);
                    }
                }

                /*
                |--------------------------------------------------------------------------
                | if not marked read, mark it as read
                |--------------------------------------------------------------------------
                */

                if (NotificationEnum::META_IS_READ_NO == $model -> getMetaIsRead())
                {
                    $model -> update(
                        array(
                            NotificationTable::META_IS_READ => NotificationEnum::META_IS_READ_YES,
                        )
                    );

                    $model -> save();

                    /*
                    |--------------------------------------------------------------------------
                    | let's mark it unread. wot? we're not saving changes here. marking it unread
                    | make sure that client receive it has unread so it can boldify title or show 
                    | an unread badge or something to differentiate from read notifcations if it
                    | wants
                    |--------------------------------------------------------------------------
                    */

                    $model -> update(
                        array(
                            NotificationTable::META_IS_READ => NotificationEnum::META_IS_READ_NO,
                        )
                    );
                }
                else
                {
                    /*
                    |--------------------------------------------------------------------------
                    | if transient delete it
                    |--------------------------------------------------------------------------
                    */
                    if (NotificationEnum::META_IS_TRANSIENT_YES == $model -> getMetaIsTransient())
                    {
                        $model -> delete();

                        // don't include it in response
                        throw new BreakException();
                    }
                }
            });

            /*
            |--------------------------------------------------------------------------
            | process dependencies
            |--------------------------------------------------------------------------
            */

            self::processDependencies();

            /*  
            |--------------------------------------------------------------------------
            | add user ids to fetch
            |--------------------------------------------------------------------------
            */

            $postsContainer = self::$dataDock -> getContainer(PostTable::getTableName());

            foreach ($postsContainer -> getDataMaps() as $postDataMap)
            {
                self::addDependency(UserTable::getTableName(), $postDataMap[PostTable::getOwnerAttribute()]);
            }

            /*
            |--------------------------------------------------------------------------
            | process newly added dependencies
            |--------------------------------------------------------------------------
            */

            self::processDependencies();

            /*
            |--------------------------------------------------------------------------
            | additional data | user follow maps
            | -------------------------------------------------------------------------
            | help build follow button
            |--------------------------------------------------------------------------
            */

            $usersContainer = self::$dataDock -> getContainer(UserTable::getTableName());

            self::fetchModelsAndAddAsAdditional(
                UserFollowTable::getTableName(),
                array(
                    UserFollowTable::FOLLOWED_USER_ID    => $usersContainer -> getIds(),
                    UserFollowTable::FOLLOWED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );

            /*
            |--------------------------------------------------------------------------
            | additional data | post likes maps
            | -------------------------------------------------------------------------
            | help build like button
            |--------------------------------------------------------------------------
            */

            $postsContainer = self::$dataDock -> getContainer(PostTable::getTableName());

            self::fetchModelsAndAddAsAdditional(
                PostLikeTable::getTableName(),
                array(
                    PostLikeTable::LIKED_POST_ID    => $postsContainer -> getIds(),
                    PostLikeTable::LIKED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );

            /*
            |--------------------------------------------------------------------------
            | additional data | post save maps
            | -------------------------------------------------------------------------
            | help build post save button
            |--------------------------------------------------------------------------
            */

            self::fetchModelsAndAddAsAdditional(
                PostSaveTable::getTableName(),
                array(
                    PostSaveTable::SAVED_POST_ID    => $postsContainer -> getIds(),
                    PostSaveTable::SAVED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );
        });
    }
}
